function refreshFriendsList()
{
	var username = getUsername();
	debug("Getting friends list for " + username);
	var url = AUDIOSCROBBLER_WS_URL + "/user/" + username + "/friends.xml";
	Xml.makeRequest(url, null, refreshFriendsListCallback, null);
	friendsList.clearList();
	friendsList.displayMessage("Fetching friends list ...");
}

function refreshFriendsListCallback(xmlRequest)
{
	friendsList.clearList();
	var friends = parseFriendsList(xmlRequest);
	
	if (friends == ERROR_CONNECTION_PROBLEM)
	{
		friendsList.displayMessage("Sorry, but I can't seem to get " + pluraliseName(getUsername()) + " friends list. If you're already connected to the internet then there may be a problem with the Audioscrobbler web service.");
		return false;
	}
	
	if (friends == null)
	{
		friendsList.displayMessage(ERROR_USER_DOES_NOT_EXIST);
		return false;
	}
	
	friendsList.addItem(getUsername(), getUsername());
	
	for (var i = 0; i < friends.length; i++)
	{
		friendsList.addItem(friends[i], friends[i]);
	}
	
	friendsList.setSelectedItem(getTracksUsername());
}

function parseFriendsList(xmlRequest)
{
	var xml = xmlRequest.responseXML;
	var friends = new Array();
	
	if (xmlRequest.status != 200)
	{
		return ERROR_CONNECTION_PROBLEM;
	}

	var rootNode = Xml.findChild(xml, "friends");
	if (!rootNode)
	{
		return null;
	}
	var friendNodes = rootNode.getElementsByTagName("user");
	
	for (var i = 0; i < friendNodes.length; i++)
	{
		var friendUsername = friendNodes[i].getAttribute("username");
		var username = getUsername();
		if (friendUsername.toLowerCase() != username.toLowerCase())
		{
			friends[friends.length] = friendUsername;
		}
	}
	
	return friends;
}

function refreshTrackList(newUser)
{
	var username = getTracksUsername();
	debug("Getting track list for " + username);
	if (newUser)
	{
		var title = document.getElementById("title");
		var titleMsg = pluraliseName(username) + " recent tracks";
		title.innerHTML = titleMsg;
		
		var tracksDiv = document.getElementById("tracks");
		Xml.removeAllChildren(tracksDiv);
		var p = document.createElement("p");
		p.appendChild(document.createTextNode("Fetching recent tracks ..."));
		tracksDiv.appendChild(p);
	}
	
	if (LOCAL_MODE)
	{
		var url = "recenttracks.xml";
	}
	else
	{
		var url = AUDIOSCROBBLER_WS_URL + "/user/" + username + "/recenttracks.xml";
	}
	Xml.makeRequest(url, null, refreshTrackListCallback, null);
	
	if (tracksUpdaterTimeout)
	{
		window.clearTimeout(tracksUpdaterTimeout);
	}
	tracksUpdaterTimeout = window.setTimeout("refreshTrackList()", TRACKS_REFRESH_TIME);
}

function refreshTrackListCallback(xmlRequest)
{
	var tracks = parseLatestTracks(xmlRequest);
	debug("got " + tracks.length + " tracks");
	var tracksDiv = document.getElementById("tracks");
	Xml.removeAllChildren(tracksDiv);
	
	var currentDate = new Date();
	
	if (tracks == ERROR_CONNECTION_PROBLEM)
	{
		var p = document.createElement("p");
		var text = document.createTextNode("Sorry, but I couldn't get " + pluraliseName(getTracksUsername()) + " recent tracks. If you are already connected to the internet, then there may be a problem with the Audioscrobbler webservice.");
		p.appendChild(text);
		tracksDiv.appendChild(p);
	}
	else if (tracks.length == 0)
	{
		var p = document.createElement("p");
		var text = document.createTextNode("No recently played tracks.");
		p.appendChild(text);
		tracksDiv.appendChild(p);
	}
	else
	{
		tracksDiv.style.visibility = "hidden";
		var trackList = document.createElement("ol");
		trackList.id = "trackList";
		for (var i = 0; i < tracks.length; i++)
		{
			var track = tracks[i];
			var listEntry = document.createElement("li");
			var playedTime = document.createElement("div");
			playedTime.className = PLAYED_TIME_CLASS_NAME;
			playedTimeText = document.createTextNode(dateToTimeString(track.date));
			playedTime.appendChild(playedTimeText);
			listEntry.appendChild(playedTime);
			var trackDetails = document.createElement("div");
			trackDetails.className = TRACK_DETAILS_CLASS_NAME;
			trackDetails.onclick = createLinkHandler(track.url);
			var trackDetailsText = document.createTextNode(track.artist + " - " + track.title);
			trackDetails.appendChild(trackDetailsText);
			listEntry.appendChild(trackDetails);
			trackList.appendChild(listEntry);
		}
		tracksDiv.appendChild(trackList);
		window.setTimeout("fixTrackListHeight()", 1);
	}
}

function fixTrackListHeight()
{
	var tracksDiv = document.getElementById("tracks");
	debug("tracksDiv.offsetHeight == " + tracksDiv.offsetHeight);
	if (typeof(tracksDiv.offsetHeight) == "undefined")
	{
		// The front is not shown so the div has no height
		trackListHeightNeedsFixing = true;
		return;
	}
	if (tracksDiv.offsetHeight > 184)
	{
		debug("Removing last entry");
		// Remove one entry from the bottom of the list
		var trackList = document.getElementById("trackList");
		trackList.removeChild(trackList.lastChild);
		window.setTimeout("fixTrackListHeight()", 1);
	}
	else
	{
		tracksDiv.style.visibility = "visible";
	}
}

function parseLatestTracks(xmlRequest)
{

	var xml = xmlRequest.responseXML;
	
	var tracks = new Array();
	if (xmlRequest.status != 200)
	{
		return ERROR_CONNECTION_PROBLEM;
	}
	var rootNode = Xml.findChild(xml, "recenttracks");
	if (!rootNode)
	{
		alert(xmlRequest.responseText);
		return tracks;
	}
	
	var trackNodes = rootNode.getElementsByTagName("track");
	
	for (var i = 0; i < trackNodes.length; i++)
	{
		var trackNode = trackNodes[i];
		var tempNode;
		
		tempNode = Xml.findChild(trackNode, "artist");
		var artist = tempNode.firstChild.nodeValue;
		tempNode = Xml.findChild(trackNode, "name");
		var title = tempNode.firstChild.nodeValue;
		tempNode = Xml.findChild(trackNode, "url");
		var url = tempNode.firstChild.nodeValue;
		tempNode = Xml.findChild(trackNode, "date");
		var date = new Date(0);
		
		date.setTime((tempNode.getAttribute("uts") - 0) * 1000);
		
		tracks[i] =
		{
			artist: artist,
			title: title,
			url: url,
			date: date
		}
	}
	
	return tracks;
}

function doVersionCheck()
{
	debug("Checking for new version");
	Xml.makeRequest(VERSION_CHECK_URL, null, doVersionCheckCallback, null);
}

function doVersionCheckCallback(xmlRequest)
{
	versionDetails = parseVersionXml(xmlRequest);

	debug("Current version is " + VERSION);
	debug("Server version is " + versionDetails.version);

	if (versionDetails.version > VERSION)
	{
		// A newer version is available
		debug("Found a new version");
		var newVersionAvailableField = document.getElementById(NEW_VERSION_AVAILABLE_FIELD);
		addEvent(newVersionAvailableField, "click", versionDetails.showPage, false);
		Dhtml.revealElement(newVersionAvailableField);
	}
	else
	{
		debug("No new version found");
		Dhtml.hideElement(NEW_VERSION_AVAILABLE_FIELD);
	}
}

function parseVersionXml(xmlRequest)
{
	var versionDetails = {version: 0, showPage: null};

	if (xmlRequest.status == 200)
	{
		var rootNode = Xml.findChild(xmlRequest.responseXML, "application");
		versionDetails.version = parseFloat(rootNode.getAttribute("version"));
		var url = rootNode.getAttribute("url");
		versionDetails.showPage = createLinkHandler(url);
	}
	
	return versionDetails;
}

function createLinkHandler(url)
{
	return function()
	{
		widget.openURL(url);
	}
}

function dateToTimeString(date)
{
	var hours = date.getHours();
	var minutes = date.getMinutes();
	
	hours = zeroPad(hours, 2);
	minutes = zeroPad(minutes, 2);
	
	return hours + ":" + minutes;
}

function zeroPad(num, digits)
{
	num = num + "";
	var zerosRequired = digits - num.length;
	for (var i = 0; i < zerosRequired; i++)
	{
		num = "0" + num;
	}
	return num;
}

function pluraliseName(name)
{
	name += "'";
	if (name.substr(-1, 1).toLowerCase() != "s")
	{
		name += "s";
	}
	return name;
}

function debug(msg)
{
	if (DEBUG_MODE)
	{
		alert(msg);
	}
}